<?php

use Modules\Courses\Services\CourseService;



/**
 * Upload base64 image into custom storage folder.
 *
 * @param string $dirName   Required. Directory name
 * @param string $imageUrl  Required. Base64 image string
 * @return array The process of array record
 */
if (! function_exists('uploadBase64Image')) {

    function uploadBase64Image($dirName, $name, $imageUrl)
    {

        $file_ext       = ".png";
        $directoryUrl   = storage_path('app/public/' . $dirName);
        $i = 0;
        while (file_exists($directoryUrl . '/' . $name . $file_ext)) {
            $i++;
            $name = $name . "(" . $i . ")";
        }

        $fileName           = $name . $file_ext;
        $path               = $directoryUrl . '/' . $fileName;
        // Check if directory exists, if not create it recursively
        if (!is_dir($directoryUrl)) {
            if (!mkdir($directoryUrl, 0755, true)) {
                throw new \Exception("Failed to create directory: $directoryUrl");
            }
        }

        // Decode the base64 image
        $image = base64_decode(preg_replace('#^data:image/\w+;base64,#i', '', $imageUrl));
        if ($image === false) {
            throw new \Exception("Failed to decode base64 image");
        }

        // Save the image file
        $storeFile = file_put_contents($path, $image);
        if ($storeFile === false) {
            throw new \Exception("Failed to save image file: $path");
        }

        if ($storeFile) {
            return $dirName . '/' . $fileName;
        }

        return '';
    }

    /**
     *return pagination select options
     *
     * @return response()
     */
    if (!function_exists('perPageOpt')) {

        function perPageOpt()
        {

            return [10, 20, 30, 50, 100, 200];
        }
    }

    /**
     * Get course duration in hours and minutes 
     * @param int $contentLength
     * @return string
     */
    if (!function_exists('getCourseDuration')) {
        function getCourseDuration($contentLength)
        {
            if ($contentLength == 0) {
                return '';
            }

            $hours = floor($contentLength / 3600);
            $minutes = floor(($contentLength % 3600) / 60);
            $seconds = $contentLength % 60;

            $duration = '';

            if ($hours > 0) {
                $duration .= $hours . ' ' . ($hours > 1 ? __('courses::courses.hrs') : __('courses::courses.hr'));
            }

            if ($minutes > 0) {
                $duration .= ($duration ? ' : ' : '') . $minutes . ' ' . ($minutes > 1 ? __('courses::courses.mins') : __('courses::courses.min'));
            }

            if ($duration === '') {
                $duration .= $seconds . ' ' . __('courses::courses.sec') ;
            }

            return $duration;
        }
    }

    if(!function_exists('courseMenuOptions')) {
        function courseMenuOptions($role)
        {
            switch ($role) {
                case 'tutor':
                    return [
                        [
                            'tutorSortOrder' => 4,
                            'route' => 'courses.tutor.courses',
                            'onActiveRoute' => ['courses.tutor.courses', 'courses.tutor.edit-course', 'courses.tutor.create-course'],
                            'title' => __('courses::courses.manage_courses'),
                            'icon'  => '<i class="am-icon-book-1"></i>',
                            'accessibility' => ['tutor'],
                            'disableNavigate' => true,
                        ]
                    ];
                    break;
                case 'student':
                    return [
                        [
                            'route' => 'courses.course-list',
                            'studentSortOrder' => 3,
                            'onActiveRoute' => ['courses.course-list'],
                            'title' => __('courses::courses.my_learning'),
                            'icon'  => '<i class="am-icon-book-1"></i>',
                            'accessibility' => ['student'],
                            'disableNavigate' => true,
                        ],
                        [
                            'route' => 'courses.search-courses',
                            'studentSortOrder' => 5,
                            'onActiveRoute' => ['courses.search-courses'],
                            'title' => __('courses::courses.find_courses'),
                            'icon'  => '<i class="am-icon-book"></i>',
                            'accessibility' => ['student'],
                            'disableNavigate' => true,
                        ],
                    ];
                    break;
                case 'admin':
                    return [
                        [
                            'title' =>  __('courses::courses.manage_courses'),
                            'icon'  => 'icon-book-open',
                            'routes' => [
                                'courses.admin.courses' => __('courses::courses.all_courses'),
                                'courses.admin.categories' => __('courses::courses.categories'),
                                'courses.admin.course-enrollments' => __('courses::courses.course_enrollments'),
                                'courses.admin.commission-setting' => __('courses::courses.commission_settings'),
                            ],
                        ]
                    ];
                    break;
                default:
                    return [];
            }
        }
    }

    if(!function_exists('getFeaturedCourses')) {
        function getFeaturedCourses($userId = null)
        {
            return (new CourseService())->getFeaturedCourses($userId);
        }
    }
}
