<?php

namespace Modules\Courses\Livewire\Pages\Student\CourseTaking;

use Modules\Courses\Services\CourseService;
use Modules\Courses\Services\CurriculumService;
use Illuminate\Support\Facades\Auth;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Url;
use Livewire\Component;

class CourseTaking extends Component
{

    public $logo;
    public $activeCurriculum;
    public $instructorCoursesCount;
    public $slug;
    public $rating;
    public $description;
    public $isLoading = true;
    public $progress;
    public $studentRating;
    public $role;
    public $backRoute = null;
    public $curriculumOrder = [];

    public $socialIcons = [
        'Facebook' => 'am-icon-facebook-1',
        'X/Twitter'=> 'am-icon-twitter-02',
        'LinkedIn' => 'am-icon-linkedin-02',
        'Instagram' => 'am-icon-instagram',
        'Pinterest' => 'am-icon-pinterest',
        'Youtube' => 'am-icon-youtube',
        'TikTok' => 'am-icon-tiktok-02',
        'WhatsApp' => 'am-icon-whatsapp',
    ];

    #[Url] 
    public $redirect = 'courses-list';

    public function mount() {
        $this->slug = request()->route('slug');
        $this->role = auth()?->user()?->role;
        $logo = setting('_general.logo_white');
        $this->logo = !empty($logo[0]['path']) ? asset('storage/'.$logo[0]['path']) : asset('modules/courses/images/logo.svg');

        if(!$this->course){
            abort(404);
        }

        if($this->role == 'tutor' && $this->course?->instructor_id != auth()?->user()?->id) {
            return $this->redirect(route('courses.course-detail', ['slug' => $this->course->slug]));
        }
    
        $courseAddedToStudent = (new CourseService())->getStudentCourse($this->course->id);

        if($this->role == 'student' && !$courseAddedToStudent) {
            return $this->redirect(route('courses.search-courses'));
        }

        if(!empty($this->course->course_watchtime_sum_duration) && !empty($this->course->content_length)) {
            $progress = ($this->course->course_watchtime_sum_duration / $this->course->content_length) * 100;
            $this->progress = $progress >= 99 ? 100 : $progress;
        }

        $firstCurriculum = $this->course?->sections?->first()?->curriculums?->first();
        if($firstCurriculum) {
            $this->activeCurriculum = $firstCurriculum->toArray();
        }
        
        $this->studentRating = $this->course->ratings->where('student_id', auth()->id())->first();

        if($this->role == 'student'){
            $this->backRoute = $this->redirect == 'courses-list' ? route('courses.course-list') : route('courses.course-detail', ['slug' => $this->course->slug]);
        }

        if($this->role == 'admin'){
            $this->backRoute =  route('courses.admin.courses');
        }
        
        if($this->course->instructor_id == Auth::id() ){
            $this->backRoute =  route('courses.tutor.courses');
        }

        // Build curriculum order array for navigation
        $allCurriculums = collect();
        foreach ($this->course->sections as $section) {
            $allCurriculums = $allCurriculums->concat($section->curriculums);
        }

        $sortedCurriculums = $allCurriculums->values();
        
        for ($i = 0; $i < $sortedCurriculums->count(); $i++) {
            $currentCurriculum = $sortedCurriculums[$i];
            $nextItem = $sortedCurriculums[$i + 1] ?? null;
            
            $this->curriculumOrder[$currentCurriculum->id] = $nextItem ? $nextItem->id : null;
        }

    }

    #[Computed(persist: true)]
    public function course(){
        return (new CourseService())->getCourse(
            slug: $this->slug,
            relations: [
                'category',
                'instructor',
                'instructor.languages',
                'instructor.profile:id,user_id,first_name,last_name,image,slug,native_language,description,verified_at',
                'instructor.socialProfiles',
                'instructor.address',
                'subCategory',
                'language',
                'thumbnail',
                'promotionalVideo',
                'pricing',
                'noticeboards',
                'sections' => function($query) {
                    $query->withWhereHas('curriculums', function($subQuery) {
                        $subQuery->whereNotNull('media_path')->orWhereNotNull('article_content');
                        $subQuery->with('watchtime');
                        $subQuery->orderBy('sort_order', 'asc');
                    });
                },
                'ratings.student.profile',
                'ratings.student.address',
            ],
            withSum: [
                'courseWatchtime' => 'duration'  // Add this line to get sum of duration
            ],
            withAvg: [
                'ratings' => 'rating',
            ],
            withCount: [
                'ratings', 
                'sections', 
                'curriculums', 
                'instructorReviews', 
                'faqs', 
                'enrollments',
                'instructor as active_students_count' => function($query) {  // Count total enrollments
                    $query->withCount('courses')
                        ->join( config('courses.db_prefix').'courses', 'users.id', '=', config('courses.db_prefix').'courses.instructor_id')
                        ->join( config('courses.db_prefix').'enrollments', config('courses.db_prefix').'courses.id', '=', config('courses.db_prefix').'enrollments.course_id');
                },
            ],
            status: null
        );
    }

    #[Computed]
    public function totalArticles() {
        $count = 0;
        foreach($this->course->sections as $section) {
            $count += count($section->curriculums->where('type', 'article'));
        }
        return $count;
    }

    #[Computed]
    public function totalVideos() {
        $count = 0;
        foreach($this->course->sections as $section) {
            $count += count($section->curriculums->where('type', 'video'));
        }
        return $count;
    }

    public function loadCourseData() {}

    public function render()
    {
        $totalVideos = $this->totalVideos;
        $this->instructorCoursesCount       = (new CourseService())->getInstructorCoursesCount($this->course->instructor_id);

        return view('courses::livewire.student.course-taking.course-taking', ['course' => $this->course, 'totalArticles' => $this->totalArticles, 'totalVideos' => $totalVideos])->extends('courses::layouts.app');
    }

    public function loadData() {
        $this->isLoading = false;
    }

    public function setActiveCurriculum($curriculum) {
        $this->activeCurriculum = $curriculum;
    }
    public function nextCurriculum($id) {
        $nextCurriculum = (new CourseService())->getCurriculumById($id);
       
        if($nextCurriculum) {
            $this->activeCurriculum = $nextCurriculum->toArray();
        }
    }
    public function markAsCompleted() {
        $response = isDemoSite();
        if( $response ){
            $this->dispatch('showAlertMessage', type: 'error', title:  __('general.demosite_res_title') , message: __('general.demosite_res_txt'));
            return;
        }
        $curriculumId = (int)$this->activeCurriculum['id'] ?? 0;
        $sectionId = (int) $this->activeCurriculum['section_id'] ?? 0;
        $totalDuration = (int) $this->activeCurriculum['content_length'] ?? 0;
        $watchtime = (new CurriculumService())->getWatchtime($curriculumId, $sectionId);
        if($watchtime) {
            (new CurriculumService())->updateWatchtime((int) $curriculumId, (int) $sectionId, (int) $totalDuration);
        } else {
            (new CurriculumService())->addWatchtime((int) $this->course?->id, (int) $curriculumId, (int) $sectionId, (int) $totalDuration);
        }

        $courseDuration = (new CourseService())->getCourse(
            courseId: $this->course->id,
            withSum: [
                'courseWatchtime' => 'duration'
            ]
        );

        if(!empty($courseDuration->course_watchtime_sum_duration) && !empty($this->course->content_length)) {
            $this->progress = ($courseDuration->course_watchtime_sum_duration / $this->course->content_length) * 100;        
        }
        
        $this->activeCurriculum['watchtime']['duration'] = $totalDuration;
        
    }

    public function updateWatchtime() {
        $response = isDemoSite();
        if( $response ){
            return;
        }
        $curriculumId = (int)$this->activeCurriculum['id'] ?? 0;
        $sectionId = (int) $this->activeCurriculum['section_id'] ?? 0;
        $totalDuration = (int) $this->activeCurriculum['content_length'] ?? 0;

        $watchtime = (new CurriculumService())->getWatchtime($curriculumId, $sectionId);
        if($watchtime) {
            $duration = $watchtime->duration;
            if($duration < $totalDuration) {
                $updateDuration = $duration + 60 <= $totalDuration ? $duration + 60 : $totalDuration;
                (new CurriculumService())->updateWatchtime((int) $curriculumId, (int) $sectionId, (int) $updateDuration);
            }
        } else {
            $updateDuration = $totalDuration > 60 ? 60 : $totalDuration;
            (new CurriculumService())->addWatchtime((int) $this->course?->id, (int) $curriculumId, (int) $sectionId, (int) $updateDuration);
        }
        // Refresh course data to get updated watchtime values
        $courseDuration = (new CourseService())->getCourse(
            courseId: $this->course->id,
            withSum: [
                'courseWatchtime' => 'duration'
            ]
        );

        if(!empty($courseDuration->course_watchtime_sum_duration) && !empty($this->course->content_length)) {
            $this->progress = ($courseDuration->course_watchtime_sum_duration / $this->course->content_length) * 100;        
        }
    }

    public function submitRating() {
        $response = isDemoSite();
        if( $response ){
            $this->dispatch('showAlertMessage', type: 'error', title:  __('general.demosite_res_title') , message: __('general.demosite_res_txt'));
            return;
        }
        $data = $this->validate([
            'rating' => 'required|numeric|min:1|max:5',
            'description' => 'required|string|max:1000',
        ]);

        $isAdded = (new CourseService())->addCourseRating($this->course->id, $data);
        if($isAdded) {
            $this->dispatch('showAlertMessage', type: 'success', title: __('courses::courses.rating_added'), message: __('courses::courses.rating_added_successfully'));
        } else {
            $this->dispatch('showAlertMessage', type: 'error', title: __('courses::courses.error_title'), message: __('courses::courses.rating_add_failed'));
        }

        $courseRatings = (new CourseService())->getCourse(
            courseId: $this->course->id,
            relations: ['ratings']
        );

        $this->studentRating = $courseRatings->ratings->where('student_id', auth()->id())->first();
    }
}
